<?php
/*
Plugin Name: FanaticoWeb Promo Manager (Free)
Description: Gestione promozioni automatiche per WooCommerce con sconto percentuale su categorie selezionate.
Version: 1.1.0
Author: FanaticoWeb.com Solutions Srl
Author URI: https://www.fanaticoweb.com
Text Domain: fanaticoweb-promo-manager
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class FanaticoWeb_Promo_Manager_Free {

    const LOGO_FULL  = 'https://www.fanaticoweb.com/wp-content/uploads/2024/12/logo-white.webp';

    public function __construct() {

        // Admin
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );

        // Frontend (sconto carrello)
        add_action( 'woocommerce_cart_calculate_fees', array( $this, 'apply_cart_discount' ), 20, 1 );

        // Stile per il totale carrello (opzionale, ma utile)
        add_action( 'wp_head', array( $this, 'frontend_styles' ) );
    }

    /* -------------------------------------------------------------------------
     *  SETTINGS
     * ---------------------------------------------------------------------- */

    public function register_settings() {

        register_setting( 'fwpm_promo_options', 'fwpm_enabled', array(
            'type'              => 'boolean',
            'sanitize_callback' => function( $value ) {
                return $value ? 1 : 0;
            },
            'default'           => 0,
        ) );

        register_setting( 'fwpm_promo_options', 'fwpm_discount_percent', array(
            'type'              => 'number',
            'sanitize_callback' => function( $value ) {
                $value = floatval( $value );
                if ( $value < 0 )  $value = 0;
                if ( $value > 100 ) $value = 100;
                return $value;
            },
            'default'           => 20,
        ) );

        register_setting( 'fwpm_promo_options', 'fwpm_min_qty', array(
            'type'              => 'integer',
            'sanitize_callback' => function( $value ) {
                $value = intval( $value );
                return $value < 1 ? 1 : $value;
            },
            'default'           => 2,
        ) );

        register_setting( 'fwpm_promo_options', 'fwpm_date_from', array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => '',
        ) );

        register_setting( 'fwpm_promo_options', 'fwpm_date_to', array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => '',
        ) );

        register_setting( 'fwpm_promo_options', 'fwpm_categories', array(
            'type'              => 'array',
            'sanitize_callback' => function( $value ) {
                if ( ! is_array( $value ) ) {
                    $value = array();
                }
                return array_map( 'intval', $value );
            },
            'default'           => array(),
        ) );

        register_setting( 'fwpm_promo_options', 'fwpm_cart_text', array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => 'Promo - {discount}% sui prodotti selezionati',
        ) );
    }

    /* -------------------------------------------------------------------------
     *  ADMIN MENU & PAGE
     * ---------------------------------------------------------------------- */

    public function add_menu() {

        // Sotto WooCommerce nel menu admin
        add_submenu_page(
            'woocommerce',
            __( 'FanaticoWeb Promo Manager', 'fanaticoweb-promo-manager' ),
            __( 'Promo Manager (Free)', 'fanaticoweb-promo-manager' ),
            'manage_woocommerce',
            'fanaticoweb-promo-manager',
            array( $this, 'render_settings_page' )
        );
    }

    public function render_settings_page() {

        if ( ! function_exists( 'WC' ) ) {
            echo '<div class="notice notice-error"><p><strong>WooCommerce</strong> deve essere attivo per usare FanaticoWeb Promo Manager.</p></div>';
            return;
        }

        $enabled    = (bool) get_option( 'fwpm_enabled', 0 );
        $discount   = get_option( 'fwpm_discount_percent', 20 );
        $min_qty    = get_option( 'fwpm_min_qty', 2 );
        $date_from  = get_option( 'fwpm_date_from', '' );
        $date_to    = get_option( 'fwpm_date_to', '' );
        $categories = (array) get_option( 'fwpm_categories', array() );
        $text_row   = get_option( 'fwpm_cart_text', 'Promo - {discount}% sui prodotti selezionati' );

        $now     = current_time( 'timestamp' );
        $active  = $this->is_promo_currently_active();
        $status  = $active ? 'ATTIVA' : 'INATTIVA';
        $status_class = $active ? 'fwpm-status-active' : 'fwpm-status-inactive';

        ?>
        <div class="wrap fwpm-wrap">
            <style>
                .fwpm-wrap {
                    max-width: 900px;
                    font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
                }
                .fwpm-header {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    background: linear-gradient(135deg, #1E2FBF, #2C4BE6);
                    padding: 18px 22px;
                    border-radius: 16px;
                    margin: 24px 0;
                    color: #fff;
                    box-shadow: 0 15px 35px rgba(0,0,0,0.18);
                }
                .fwpm-header-left {
                    display: flex;
                    align-items: center;
                    gap: 14px;
                }
                .fwpm-header-logo-main img {
                    height: 46px;
                    width: auto;
                    display: block;
                }
                .fwpm-header-logo-small img {
                    height: 28px;
                    width: auto;
                    display: block;
                    opacity: 0.85;
                }
                .fwpm-header-text {
                    display: flex;
                    flex-direction: column;
                    gap: 2px;
                }
                .fwpm-header-title {
                    font-size: 20px;
                    font-weight: 600;
                    letter-spacing: 0.02em;
                }
                .fwpm-header-subtitle {
                    font-size: 13px;
                    opacity: 0.88;
                }
                .fwpm-header-right {
                    text-align: right;
                }
                .fwpm-badge-free {
                    display: inline-block;
                    padding: 4px 10px;
                    border-radius: 999px;
                    background: #ffffff;
                    color: #1E2FBF;
                    font-size: 11px;
                    font-weight: 600;
                    text-transform: uppercase;
                    letter-spacing: 0.08em;
                    margin-bottom: 6px;
                }
                .fwpm-status-pill {
                    display: inline-block;
                    padding: 4px 10px;
                    border-radius: 999px;
                    font-size: 11px;
                    font-weight: 600;
                }
                .fwpm-status-active {
                    background: rgba(0, 255, 163, 0.18);
                    color: #00e09a;
                    border: 1px solid rgba(0, 255, 163, 0.5);
                }
                .fwpm-status-inactive {
                    background: rgba(255, 255, 255, 0.13);
                    color: #ffffff;
                    border: 1px solid rgba(255, 255, 255, 0.3);
                }

                .fwpm-grid {
                    display: grid;
                    grid-template-columns: 1.1fr 0.9fr;
                    gap: 18px;
                    margin-top: 10px;
                }
                @media (max-width: 960px) {
                    .fwpm-grid {
                        grid-template-columns: 1fr;
                    }
                }

                .fwpm-card {
                    background: #ffffff;
                    border-radius: 14px;
                    padding: 18px 20px;
                    margin-bottom: 18px;
                    box-shadow: 0 10px 25px rgba(18, 24, 40, 0.04);
                    border: 1px solid #E2E4F0;
                }
                .fwpm-card h2 {
                    margin: 0 0 12px;
                    font-size: 15px;
                    color: #1E2FBF;
                    display: flex;
                    align-items: center;
                    gap: 8px;
                }
                .fwpm-card h2 .fwpm-card-icon {
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                    width: 22px;
                    height: 22px;
                    border-radius: 999px;
                    background: rgba(44, 75, 230, 0.08);
                    font-size: 13px;
                }
                .fwpm-field-group {
                    margin-bottom: 12px;
                }
                .fwpm-field-group label {
                    display: block;
                    font-weight: 500;
                    font-size: 13px;
                    color: #333;
                    margin-bottom: 4px;
                }
                .fwpm-field-group small.description {
                    display: block;
                    font-size: 11px;
                    color: #777;
                    margin-top: 3px;
                }
                .fwpm-toggle-row {
                    display: flex;
                    align-items: center;
                    gap: 8px;
                    margin-bottom: 10px;
                }
                .fwpm-toggle-row label {
                    margin: 0;
                    font-weight: 500;
                }
                .fwpm-inline-fields {
                    display: flex;
                    gap: 10px;
                    align-items: center;
                }
                .fwpm-inline-fields input[type="number"],
                .fwpm-inline-fields input[type="date"] {
                    max-width: 140px;
                }
                .fwpm-wrap input[type="text"],
                .fwpm-wrap input[type="number"],
                .fwpm-wrap input[type="date"],
                .fwpm-wrap select {
                    border-radius: 8px;
                    border: 1px solid #D3D6E4;
                    padding: 6px 8px;
                    font-size: 13px;
                    width: 100%;
                }
                .fwpm-wrap select[multiple] {
                    min-height: 220px;
                }
                .fwpm-footer {
                    font-size: 12px;
                    color: #666;
                    margin-top: 12px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    gap: 10px;
                }
                .fwpm-footer a {
                    color: #2C4BE6;
                    text-decoration: none;
                }
                .fwpm-footer a:hover {
                    text-decoration: underline;
                }

                .fwpm-preview-label {
                    font-family: system-ui, sans-serif;
                    font-size: 12px;
                    color: #666;
                    margin-top: 6px;
                }
                .fwpm-preview-badge {
                    display: inline-block;
                    margin-top: 3px;
                    background: #FFE8E8;
                    color: #C40000;
                    border-radius: 999px;
                    padding: 4px 10px;
                    font-size: 12px;
                    font-weight: 600;
                }

                .fwpm-submit-wrapper {
                    margin-top: 12px;
                    display: flex;
                    justify-content: flex-end;
                }
                .fwpm-submit-wrapper .button-primary {
                    background: #2C4BE6;
                    border-color: #2C4BE6;
                    border-radius: 999px;
                    padding: 0 20px;
                    height: 34px;
                    line-height: 32px;
                    box-shadow: 0 8px 18px rgba(44, 75, 230, 0.35);
                    font-weight: 600;
                }
                .fwpm-submit-wrapper .button-primary:hover {
                    background: #1E2FBF;
                    border-color: #1E2FBF;
                }
            </style>

            <div class="fwpm-header">
                <div class="fwpm-header-left">
                    <div class="fwpm-header-logo-main">
                        <img src="<?php echo esc_url( self::LOGO_FULL ); ?>" alt="FanaticoWeb Logo">
                    </div>
                    <div class="fwpm-header-text">
                        <div class="fwpm-header-title">FanaticoWeb Promo Manager</div>
                        <div class="fwpm-header-subtitle">
                            Sconti automatici direttamente nel carrello del tuo WooCommerce, plugin performante e leggero!
                        </div>
                    </div>
                </div>
                <div class="fwpm-header-right">
                    <div class="fwpm-badge-free">FREE VERSION</div><br>
                    <span class="fwpm-status-pill <?php echo esc_attr( $status_class ); ?>">
                        Stato promo: <?php echo esc_html( $status ); ?>
                    </span>
                </div>
            </div>

            <form method="post" action="options.php">
                <?php settings_fields( 'fwpm_promo_options' ); ?>

                <div class="fwpm-grid">
                    <!-- COLONNA SINISTRA -->
                    <div>
                        <div class="fwpm-card">
                            <h2>
                                <span class="fwpm-card-icon">⚙️</span>
                                Impostazioni generali
                            </h2>

                            <div class="fwpm-field-group fwpm-toggle-row">
                                <label for="fwpm_enabled">
                                    <input type="checkbox" id="fwpm_enabled" name="fwpm_enabled" value="1" <?php checked( $enabled, true ); ?>>
                                    Abilita lo sconto automatico sul carrello
                                </label>
                            </div>

                            <div class="fwpm-field-group">
                                <label for="fwpm_discount_percent">Percentuale sconto</label>
                                <div class="fwpm-inline-fields">
                                    <input type="number" min="0" max="100" step="1" id="fwpm_discount_percent" name="fwpm_discount_percent" value="<?php echo esc_attr( $discount ); ?>">
                                    <span>%</span>
                                </div>
                                <small class="description">Inserisci la percentuale di sconto da applicare ai prodotti in promozione.</small>
                            </div>

                            <div class="fwpm-field-group">
                                <label for="fwpm_min_qty">Quantità minima articoli promo</label>
                                <input type="number" min="1" id="fwpm_min_qty" name="fwpm_min_qty" value="<?php echo esc_attr( $min_qty ); ?>">
                                <small class="description">Numero minimo di articoli in promozione presenti nel carrello per attivare lo sconto.</small>
                            </div>
                        </div>

                        <div class="fwpm-card">
                            <h2>
                                <span class="fwpm-card-icon">🗓️</span>
                                Programmazione promozione
                            </h2>

                            <div class="fwpm-field-group">
                                <label>Periodo promozione</label>
                                <div class="fwpm-inline-fields">
                                    <div>
                                        <small>Dal</small>
                                        <input type="date" id="fwpm_date_from" name="fwpm_date_from" value="<?php echo esc_attr( $date_from ); ?>">
                                    </div>
                                    <div>
                                        <small>Al</small>
                                        <input type="date" id="fwpm_date_to" name="fwpm_date_to" value="<?php echo esc_attr( $date_to ); ?>">
                                    </div>
                                </div>
                                <small class="description">
                                    Lascia vuoto per nessun limite di data. L'orario considerato è quello del sito (<?php echo esc_html( get_option( 'timezone_string' ) ?: 'WordPress' ); ?>).
                                </small>
                            </div>
                        </div>
                    </div>

                    <!-- COLONNA DESTRA -->
                    <div>
                        <div class="fwpm-card">
                            <h2>
                                <span class="fwpm-card-icon">🗂️</span>
                                Categorie in promozione
                            </h2>

                            <div class="fwpm-field-group">
                                <label for="fwpm_categories">Seleziona le categorie prodotto</label>
                                <select id="fwpm_categories" name="fwpm_categories[]" multiple>
                                    <?php
                                    $terms = get_terms( array(
                                        'taxonomy'   => 'product_cat',
                                        'hide_empty' => false,
                                    ) );

                                    if ( ! is_wp_error( $terms ) ) {
                                        foreach ( $terms as $term ) {
                                            printf(
                                                '<option value="%1$d" %2$s>%3$s (ID: %1$d)</option>',
                                                intval( $term->term_id ),
                                                selected( in_array( $term->term_id, $categories, true ), true, false ),
                                                esc_html( $term->name )
                                            );
                                        }
                                    }
                                    ?>
                                </select>
                                <small class="description">
                                    Seleziona le categorie da includere. Le sottocategorie verranno incluse automaticamente.
                                </small>
                            </div>
                        </div>

                        <div class="fwpm-card">
                            <h2>
                                <span class="fwpm-card-icon">🧾</span>
                                Testo sconto nel carrello
                            </h2>

                            <div class="fwpm-field-group">
                                <label for="fwpm_cart_text">Testo riga sconto carrello</label>
                                <input type="text" id="fwpm_cart_text" name="fwpm_cart_text" value="<?php echo esc_attr( $text_row ); ?>">
                                <small class="description">
                                    Puoi usare <code>{discount}</code> per inserire la percentuale.<br>
                                    Esempio: <code>Black Friday -{discount}% Scarpe &amp; Felpe</code>
                                </small>

                                <?php
                                $preview_label = str_replace( '{discount}', $discount, $text_row );
                                ?>
                                <div class="fwpm-preview-label">Anteprima:</div>
                                <div class="fwpm-preview-badge">
                                    <?php echo esc_html( $preview_label ); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="fwpm-submit-wrapper">
                    <?php submit_button( 'Salva le modifiche', 'primary', 'submit', false ); ?>
                </div>

                <div class="fwpm-footer">
                    <div>
                        Promo Manager (Free) &copy; <?php echo esc_html( date( 'Y' ) ); ?> — Sviluppato da <a href="https://www.fanaticoweb.com" target="_blank" rel="noopener noreferrer">FanaticoWeb.com</a> | <a href="https://www.fanaticoweb.com/wp-content/uploads/2025/11/Liberatoria.pdf" target="_blank" rel="noopener noreferrer">Liberatoria</a>
                    </div>
                    <div>
                        Supporto tecnico: <a href="mailto:supporto@fanaticoweb.com">supporto@fanaticoweb.com</a>
                    </div>
                </div>

            </form>
        </div>
        <?php
    }

    /* -------------------------------------------------------------------------
     *  LOGICA SCONTO CARRELLO
     * ---------------------------------------------------------------------- */

    /**
     * Applica lo sconto al carrello in base alle impostazioni.
     */
    public function apply_cart_discount( $cart ) {

        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
            return;
        }

        if ( ! $cart instanceof WC_Cart ) {
            return;
        }

        $enabled = (bool) get_option( 'fwpm_enabled', 0 );
        if ( ! $enabled ) {
            return;
        }

        if ( ! $this->is_promo_in_date_range() ) {
            return;
        }

        $discount_percent = floatval( get_option( 'fwpm_discount_percent', 20 ) );
        $min_qty          = intval( get_option( 'fwpm_min_qty', 2 ) );
        $categories       = (array) get_option( 'fwpm_categories', array() );
        $label_template   = get_option( 'fwpm_cart_text', 'Promo - {discount}% sui prodotti selezionati' );

        if ( $discount_percent <= 0 || empty( $categories ) ) {
            return;
        }

        $promo_qty   = 0;
        $promo_total = 0;

        foreach ( $cart->get_cart() as $cart_item ) {

            if ( empty( $cart_item['data'] ) || ! isset( $cart_item['product_id'] ) ) {
                continue;
            }

            $product_id = $cart_item['product_id'];
            $qty        = $cart_item['quantity'];
            $line_total = $cart_item['line_total'];

            if ( $qty <= 0 || $line_total <= 0 ) {
                continue;
            }

            if ( ! $this->product_in_categories( $product_id, $categories ) ) {
                continue;
            }

            $promo_qty   += $qty;
            $promo_total += $line_total;
        }

        if ( $promo_qty < $min_qty || $promo_total <= 0 ) {
            return;
        }

        $discount_amount = $promo_total * ( $discount_percent / 100 );
        if ( $discount_amount <= 0 ) {
            return;
        }

        $label = str_replace( '{discount}', $discount_percent, $label_template );

        // Applichiamo come "fee" negativa (sconto).
        $cart->add_fee( $label, -$discount_amount, false );
    }

    /**
     * Verifica se la promo è attualmente attiva (abilitata + range date valido).
     */
    protected function is_promo_currently_active() {
        $enabled = (bool) get_option( 'fwpm_enabled', 0 );
        if ( ! $enabled ) {
            return false;
        }
        return $this->is_promo_in_date_range();
    }

    /**
     * Controlla se la data corrente è dentro al range impostato.
     */
    protected function is_promo_in_date_range() {
        $date_from = get_option( 'fwpm_date_from', '' );
        $date_to   = get_option( 'fwpm_date_to', '' );
        $now       = current_time( 'timestamp' );

        if ( $date_from ) {
            $from_ts = strtotime( $date_from . ' 00:00:00' );
            if ( $from_ts && $now < $from_ts ) {
                return false;
            }
        }

        if ( $date_to ) {
            $to_ts = strtotime( $date_to . ' 23:59:59' );
            if ( $to_ts && $now > $to_ts ) {
                return false;
            }
        }

        return true;
    }

    /**
     * Verifica se un prodotto appartiene a una delle categorie selezionate
     * (considerando anche i genitori).
     */
    protected function product_in_categories( $product_id, $allowed_cats ) {

        $allowed_cats = array_map( 'intval', (array) $allowed_cats );
        if ( empty( $allowed_cats ) ) {
            return false;
        }

        $terms = get_the_terms( $product_id, 'product_cat' );
        if ( empty( $terms ) || is_wp_error( $terms ) ) {
            return false;
        }

        foreach ( $terms as $term ) {
            $term_id = intval( $term->term_id );
            if ( in_array( $term_id, $allowed_cats, true ) ) {
                return true;
            }

            // Controllo dei genitori
            $ancestors = get_ancestors( $term_id, 'product_cat' );
            foreach ( $ancestors as $ancestor_id ) {
                if ( in_array( intval( $ancestor_id ), $allowed_cats, true ) ) {
                    return true;
                }
            }
        }

        return false;
    }

    /* -------------------------------------------------------------------------
     *  FRONTEND STYLES
     * ---------------------------------------------------------------------- */

    public function frontend_styles() {
        ?>
        <style>
            /* Evidenzia la riga di sconto nel carrello (solo fee con testo promo) */
            .woocommerce-cart .cart_totals .fee td,
            .woocommerce-cart .cart_totals .fee th {
                font-weight: 500;
            }
            .woocommerce-cart .cart_totals .fee th {
                color: #C40000;
            }
        </style>
        <?php
    }
}

new FanaticoWeb_Promo_Manager_Free();
